<?php

declare(strict_types=1);

/**
 * This file is part of LEPTON Core, released under the GNU GPL
 * Please see LICENSE and COPYING files in your package for details, specially for terms and warranties.
 * 
 * NOTICE:LEPTON CMS Package has several different licenses.
 * Please see the individual license in the header of each single file or info.php of modules and templates.
 *
 * @author          LEPTON Project
 * @copyright       2010-2025 LEPTON Project
 * @link            https://lepton-cms.org
 * @license         https://gnu.org/licenses/gpl.html
 * @license_terms   please see LICENSE and COPYING files in your package
 *
 */

/**
 *  LEPTON core-class for some essential values.
 *
 */ 
class LEPTON_basics
{
    const SYSTEM_DEFAULT_STR = 'System Default';
    /**
     *  Holds the 'singleton' instance of this class.
     *
     *  @type   object
     *
     */
    public static $instance;

    /**
     *  Return the internal instance.
     *
     *  @return object  The instance of this class.
     *
     */
    public static function getInstance(): object
    {
        if (null === static::$instance)
        {
            static::$instance = new static();
        }

        return static::$instance;
    }



    /**
     *  Get the error-levels.
     *
     *  @return array   An assoc. array with the error-levels.
     *
     */    
    static function get_errorlevels(): array
    {
        global $TEXT;
        // Create array
        $ER_LEVELS = [];

        // Add values to list
        if (isset($TEXT['SYSTEM_DEFAULT']))
        {
            $ER_LEVELS[''] = $TEXT['SYSTEM_DEFAULT'];
        }
        else
        {
            $ER_LEVELS[''] = self::SYSTEM_DEFAULT_STR;
        }
		
        $ER_LEVELS['0'] = 'E_NONE'; // standard for productive use		
        $ER_LEVELS[E_NOTICE] = 'E_NOTICE';
        $ER_LEVELS[E_WARNING] = 'E_WARNING';
        $ER_LEVELS['-1'] = 'E_EVERYTHING'; // highest level

        return $ER_LEVELS;
    }

    /**
     *  Backend-Theme specific language values
     *
     *  @param  string  $sKey Any valid key-string. Default is "".
     *  @return string        The value if the key or a warning about the missing key.
     *
     *  @notice You can use the method to "preload" the theme-language file passing NULL 
     *          to load them where you need them!
     */
    static function get_backend_translation(string $sKey = ""): string
    {
        global $TEXT, $THEME;

        $sLanguagePath_user     = THEME_PATH."/languages/".LANGUAGE.".php";
        $sLanguagePath_default  = THEME_PATH."/languages/EN.php";

        if (!isset($THEME))
        {
            /**
             *    Backend-Theme can also have additional language-files
             */
            if (file_exists($sLanguagePath_user))
            {
                require_once $sLanguagePath_user;
            }
            elseif( file_exists( $sLanguagePath_default ) )
            {
                require_once $sLanguagePath_default;
            }
            else
            {
                // avoid errors and conflicts for non existing $THEME 
                $THEME = [];
            }
        }

        if (isset($THEME[$sKey]))
        {
            return $THEME[$sKey];
        }
        elseif (isset($TEXT[$sKey]))
        {
            return $TEXT[$sKey];
        }
        else
        {
            return "** ".$sKey." (Key not found in Languages!)";
        }
    }

    /**
     *  Returns the path to the language file inside a given directory; mostly the current module.
     *
     *  @param  string  $sRootDir A path to the "look-up" dir
     *  @return string  The path to the language file, if the corresponding one to the current LANGUAGE doesn't exist the EN version will return.
     *
     */
    static function getLanguagePath(string $sRootDir = ""): string
    {
        $lang = $sRootDir."/languages/".LANGUAGE.".php";
        return (file_exists($lang))
            ? $lang
            : $sRootDir."/languages/EN.php"
            ;
    }
}
